"use client";

import { useEffect, useState, useRef } from "react";
import { useParams, useRouter } from "next/navigation";
import Image from "next/image";
import { toast } from "react-toastify";
import { apiGet, apiPost, API_ENDPOINTS } from "@/lib/api";
import { getSubdomain } from "@/lib/utils";
import { TypeFieldAnswer } from "@/lib/enums/TypeFieldAnswer";
import type { JobDetail, JobQuestion } from "../types";

type StatusApply = "apply-first" | "email-found" | "login" | "apply-after-login";

interface JobApplicationStepperProps {
  initialStep?: number;
}

export default function JobApplicationStepper({
  initialStep = 1,
}: JobApplicationStepperProps) {
  const params = useParams();
  const router = useRouter();
  const { id, slug } = params as { id: string; slug: string };

  const [currentStep, setCurrentStep] = useState(initialStep);
  const [data, setData] = useState<JobDetail | null>(null);
  const [status, setStatus] = useState<StatusApply>("apply-after-login");
  const [subdomain, setSubdomain] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);

  // Form step state
  const [form, setForm] = useState({
    first_name: "",
    last_name: "",
    email: "",
    phone: "",
    password: "",
    cover_letter: "",
  });
  const [cvFile, setCvFile] = useState<File | null>(null);
  const [cvFileName, setCvFileName] = useState("");
  const [coverFile, setCoverFile] = useState<File | null>(null);
  const [coverLetterFileName, setCoverLetterFileName] = useState("");
  const cvInputRef = useRef<HTMLInputElement | null>(null);
  const coverInputRef = useRef<HTMLInputElement | null>(null);
  const [isAgree, setIsAgree] = useState(false);

  // Questions step state
  const [questionsAnswers, setQuestionsAnswers] = useState<Record<number, any>>(
    {}
  );
  // File handling for questions
  const [questionFiles, setQuestionFiles] = useState<Record<number, File>>({});
  const [questionFileNames, setQuestionFileNames] = useState<Record<number, string>>({});
  const questionFileInputRefs = useRef<Record<number, HTMLInputElement | null>>({});

  useEffect(() => {
    const sub = getSubdomain();
    setSubdomain(sub);
    loadJob(sub);
  }, [id]);

  const loadJob = async (sub: string) => {
    if (!id) return;
    try {
      setIsLoading(false);
      const res = await apiGet<JobDetail>(
        `${API_ENDPOINTS.JobShow}/${id}`,
        undefined,
        sub
      );
      setData(res.data);
      setIsLoading(true);
    } catch (error) {
      console.error("Error loading job for apply", error);
      setIsLoading(true);
    }
  };

  // Form handlers
  const onChangeField = (key: string, value: string) => {
    setForm((prev) => ({ ...prev, [key]: value }));
  };

  const selectCV = () => {
    cvInputRef.current?.click();
  };

  const selectCoverLetter = () => {
    coverInputRef.current?.click();
  };

  const onFileCV = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      if (file.type !== "application/pdf") {
        toast.error("Only PDF files are allowed");
        return;
      }
      setCvFile(file);
      setCvFileName(file.name);
    }
  };

  const onFileCoverLetter = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      if (file.type !== "application/pdf") {
        toast.error("Only PDF files are allowed");
        return;
      }
      setCoverFile(file);
      setCoverLetterFileName(file.name);
    }
  };

  const removeCV = () => {
    setCvFile(null);
    setCvFileName("");
    if (cvInputRef.current) cvInputRef.current.value = "";
  };

  const removeCoverLetter = () => {
    setCoverFile(null);
    setCoverLetterFileName("");
    if (coverInputRef.current) coverInputRef.current.value = "";
  };

  const handleFormSubmit = () => {
    if (!isAgree) {
      toast.error("Please agree to the terms and conditions");
      return;
    }
    if (data?.job_appliction_question?.length) {
      setCurrentStep(2);
    } else {
      // Submit directly if no questions
      handleFinalSubmit();
    }
  };

  // Questions handlers
  const handleQuestionAnswer = (q: JobQuestion, value: any) => {
    setQuestionsAnswers((prev) => ({ ...prev, [q.id]: value }));
  };

  // File handling for questions
  const selectQuestionFile = (questionId: number) => {
    const input = questionFileInputRefs.current[questionId];
    if (input) {
      input.click();
    }
  };

  const handleQuestionFileSelected = (e: React.ChangeEvent<HTMLInputElement>, questionId: number) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      
      // Validate file type
      const allowedTypes = [
        'application/pdf',
        'application/msword',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'text/rtf',
        'text/plain',
        'image/jpeg',
        'image/jpg',
        'image/png',
        'image/gif'
      ];
      const maxSize = 10 * 1024 * 1024; // 10MB
      
      if (!allowedTypes.includes(file.type)) {
        toast.error('Invalid file type. Please upload PDF, Word, RTF, TXT, or image files.');
        e.target.value = '';
        return;
      }
      
      if (file.size > maxSize) {
        toast.error('File size too large. Maximum size is 10MB.');
        e.target.value = '';
        return;
      }

      // Store file
      setQuestionFiles((prev) => ({ ...prev, [questionId]: file }));
      setQuestionFileNames((prev) => ({ ...prev, [questionId]: file.name }));
      handleQuestionAnswer({ id: questionId } as JobQuestion, file);
    }
  };

  const removeQuestionFile = (questionId: number) => {
    setQuestionFiles((prev) => {
      const newFiles = { ...prev };
      delete newFiles[questionId];
      return newFiles;
    });
    setQuestionFileNames((prev) => {
      const newNames = { ...prev };
      delete newNames[questionId];
      return newNames;
    });
    handleQuestionAnswer({ id: questionId } as JobQuestion, null);
    
    // Clear input
    const input = questionFileInputRefs.current[questionId];
    if (input) {
      input.value = '';
    }
  };

  const hasQuestionFile = (questionId: number): boolean => {
    return !!questionFileNames[questionId];
  };

  const handleQuestionsSubmit = () => {
    handleFinalSubmit();
  };

  const handleFinalSubmit = async () => {
    if (isSubmitting) return;

    try {
      setIsSubmitting(true);

      // Prepare form data
      const formData = new FormData();

      // Add basic form fields - matches Angular GetValue() structure
      formData.append("job_id", id);
      formData.append("first_name", form.first_name);
      formData.append("last_name", form.last_name);
      formData.append("email", form.email);
      if (data?.company?.id) {
        formData.append("company_id", data.company.id.toString());
      }
      if (form.phone) formData.append("phone", form.phone);
      if (form.password) formData.append("password", form.password);
      // Note: cover_letter text field is separate from cover_letter file

      // Add files - matches Angular pattern
      if (cvFile) {
        formData.append("cv", cvFile, cvFile.name);
      }
      if (coverFile) {
        // Angular uses "cover_letter" for the file, not "cover_letter_file"
        formData.append("cover_letter", coverFile, coverFile.name);
      }

      // Prepare job application questions - matches Angular GetValue() logic
      // In Angular, job_appliction_question is sent as array of objects with answer property
      const jobApplicationQuestions: any[] = [];
      
      if (data?.job_appliction_question) {
        data.job_appliction_question.forEach((q) => {
          const answer = questionsAnswers[q.id];
          const questionFile = questionFiles[q.id];
          const answerType = (q.answer_type || q.type || "").toLowerCase().replace(/-/g, "_");

          // Only include questions that have answers (matches Angular checkQuestion logic)
          // For multiple_choice, answer can be empty array, so we check differently
          const hasAnswer = answerType === TypeFieldAnswer.MULTIPLE_CHOICE
            ? Array.isArray(answer) && answer.length > 0
            : answer !== undefined && answer !== null && answer !== "";

          if (hasAnswer) {
            // Build question object matching Angular JobApplicationField structure
            const questionData: any = {
              id: q.id,
              name: q.name || q.title || "",
              question: q.question || "",
              required: q.required || false,
              answer_type: answerType,
              introduction: q.introduction || null,
            };

            // For multiple_choice, set answer to null (backend expects null, not array)
            // The actual answers will be stored in job_appliction_fild_answer
            // Backend tries to insert $answer in line 126 before checking answer_type
            // So we must send null or empty string to avoid "Array to string conversion" error
            if (answerType === TypeFieldAnswer.MULTIPLE_CHOICE) {
              // Set answer to null for multiple_choice - backend will handle it separately
              // This matches Angular behavior where item.answer is null for multiple_choice
              questionData.answer = null;
              
              // Include job_appliction_fild_answer with selected answers only
              // Backend expects: array of objects with id, fild, and answer (boolean true for selected)
              // This matches Angular where question.answer is boolean for each option
              if (Array.isArray(answer) && answer.length > 0) {
                // Get all available options, ensuring we have the correct structure
                // Clean the options to remove any answer: true values from backend
                const allOptions = (q.job_appliction_fild_answer || q.options?.map((o) => ({ id: o.id, fild: o.name })) || [])
                  .map((option: any) => ({
                    id: option.id,
                    fild: option.fild || option.name, // Use fild if available, otherwise name
                  }));
                
                // Filter to only selected options and map to the format backend expects
                // Backend's storeJobApplictionAnswerMulti expects answer: true for selected options
                questionData.job_appliction_fild_answer = allOptions
                  .filter((option: any) => answer.includes(option.fild))
                  .map((option: any) => ({
                    id: option.id,
                    fild: option.fild,
                    answer: true, // Boolean true for selected options (matches Angular pattern)
                  }));
              }
            } else {
              // For other types, include the answer directly
              questionData.answer = answer;
            }

            jobApplicationQuestions.push(questionData);

            // Add question file separately - matches Angular pattern (question_file_{questionIndex})
            if (questionFile) {
              formData.append(`question_file_${q.id}`, questionFile, questionFile.name);
            }
          }
        });
      }

      // Add questions as JSON string - matches Angular pattern in GetValue()
      if (jobApplicationQuestions.length > 0) {
        formData.append("job_appliction_question", JSON.stringify(jobApplicationQuestions));
      }

      // Submit to backend
      const response = await apiPost(API_ENDPOINTS.JobApply, formData, subdomain);

      if (response.success) {
        // Success - redirect to job page or show success message
        toast.success("Application submitted successfully!");
        // Small delay to show the success message before redirect
        setTimeout(() => {
          router.push(`/job/${slug}/${id}`);
        }, 1500);
      } else {
        toast.error(response.message || "Failed to submit application. Please try again.");
      }
    } catch (error: any) {
      console.error("Error submitting application:", error);
      
      // Extract error message from response if available
      let errorMessage = "An error occurred while submitting your application. Please try again.";
      
      if (error?.message) {
        errorMessage = error.message;
      } else if (error?.response?.data?.message) {
        errorMessage = error.response.data.message;
      } else if (typeof error === "string") {
        errorMessage = error;
      }
      
      toast.error(errorMessage);
    } finally {
      setIsSubmitting(false);
    }
  };

  const getQuestionTypeName = (
    answerType: string | null | undefined
  ): string => {
    // Map enum values to display names
    const typeNames: Record<string, string> = {
      [TypeFieldAnswer.SHORT_TEXT]: "Short Text",
      [TypeFieldAnswer.YES_NO]: "Yes/No",
      [TypeFieldAnswer.SINGLE_CHOICE]: "Single Choice",
      [TypeFieldAnswer.MULTIPLE_CHOICE]: "Multiple Choice",
      [TypeFieldAnswer.DATE]: "Date",
      [TypeFieldAnswer.NUMBER]: "Number",
      [TypeFieldAnswer.FILE]: "File Upload",
    };
    
    return typeNames[answerType || ""] || answerType || "";
  };

  if (!isLoading || !data) {
    return (
      <div className="flex min-h-screen items-center justify-center">
        <div className="text-center">
          <div className="loading-spinner"></div>
          <p className="mt-4 text-gray-600">Loading...</p>
        </div>
      </div>
    );
  }

  const totalSteps = data.job_appliction_question && data.job_appliction_question.length > 0 ? 2 : 1;
  const hasQuestions = data.job_appliction_question && data.job_appliction_question.length > 0;

  return (
    <div className="m-page apply-form-page">
      <div className="row">
        <div className="col-lg-12 px-lg-5 px-sm-0 col-sm-12">
          {/* Stepper Header */}
          <div className="stepper-container mb-5">
            <div className="stepper-wrapper">
              {/* Step 1 */}
              <div className="stepper-step">
                <div className="stepper-step-content">
                  <div
                    className={`stepper-circle ${
                      currentStep >= 1 ? "active" : ""
                    }`}
                  >
                    1
                  </div>
                  <div className="stepper-label">Interested?</div>
                </div>
              </div>

              {/* Connector Line */}
              {hasQuestions && (
                <>
                  <div className="stepper-connector"></div>

                  {/* Step 2 */}
                  <div className="stepper-step">
                    <div className="stepper-step-content">
                      <div
                        className={`stepper-circle ${
                          currentStep >= 2 ? "active" : ""
                        }`}
                      >
                        {currentStep >= 2 ? (
                          <svg
                            width="20"
                            height="20"
                            viewBox="0 0 24 24"
                            fill="none"
                            xmlns="http://www.w3.org/2000/svg"
                          >
                            <path
                              d="M11 4H4C3.46957 4 2.96086 4.21071 2.58579 4.58579C2.21071 4.96086 2 5.46957 2 6V20C2 20.5304 2.21071 21.0391 2.58579 21.4142C2.96086 21.7893 3.46957 22 4 22H18C18.5304 22 19.0391 21.7893 19.4142 21.4142C19.7893 21.0391 20 20.5304 20 20V13"
                              stroke="currentColor"
                              strokeWidth="2"
                              strokeLinecap="round"
                              strokeLinejoin="round"
                            />
                            <path
                              d="M18.5 2.5C18.8978 2.10217 19.4374 1.87868 20 1.87868C20.5626 1.87868 21.1022 2.10217 21.5 2.5C21.8978 2.89782 22.1213 3.43739 22.1213 4C22.1213 4.56261 21.8978 5.10217 21.5 5.5L12 15L8 16L9 12L18.5 2.5Z"
                              stroke="currentColor"
                              strokeWidth="2"
                              strokeLinecap="round"
                              strokeLinejoin="round"
                            />
                          </svg>
                        ) : (
                          "2"
                        )}
                      </div>
                      <div className="stepper-label">Pre-screen</div>
                    </div>
                  </div>
                </>
              )}
            </div>
          </div>

          {/* Step 1: Form */}
          {currentStep === 1 && (
            <div className="border-light-gray b-r p-3">
            

              <h3 className="mb-4">
                <b>Interested?</b>
              </h3>

              <form>
                <div className="row">
                  {/* Email */}
                  <div className="col-lg-12 col-sm-12 mt-4">
                    <label htmlFor="email" className="color-p mb-1">
                      <b>Email</b>
                    </label>
                    <span className="ms-2">*</span>
                    <div className="position-relative">
                      <input
                        type="email"
                        className="form-control height"
                        id="email"
                        placeholder="Your email"
                        value={form.email}
                        onChange={(e) => onChangeField("email", e.target.value)}
                      />
                      <div className="email-error-icon">
                        <div className="email-error-square">
                          <svg
                            width="12"
                            height="12"
                            viewBox="0 0 12 12"
                            fill="none"
                            xmlns="http://www.w3.org/2000/svg"
                          >
                            <circle cx="6" cy="3" r="1" fill="white" />
                            <circle cx="6" cy="6" r="1" fill="white" />
                            <circle cx="6" cy="9" r="1" fill="white" />
                          </svg>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* First Name */}
                  <div className="col-lg-6 col-sm-12 mt-4">
                    <label htmlFor="first_name" className="color-p mb-1">
                      <b>First name</b>
                    </label>
                    <span className="ms-2">*</span>
                    <input
                      type="text"
                      className="form-control height"
                      id="first_name"
                      placeholder="First name"
                      value={form.first_name}
                      onChange={(e) =>
                        onChangeField("first_name", e.target.value)
                      }
                    />
                  </div>

                  {/* Last Name */}
                  <div className="col-lg-6 col-sm-12 mt-4">
                    <label htmlFor="last_name" className="color-p mb-1">
                      <b>Last name</b>
                    </label>
                    <span className="ms-2">*</span>
                    <input
                      type="text"
                      className="form-control height"
                      id="last_name"
                      placeholder="Last name"
                      value={form.last_name}
                      onChange={(e) =>
                        onChangeField("last_name", e.target.value)
                      }
                    />
                  </div>
                </div>

                {/* Documents */}
                <div className="row mt-6">
                  <div className="col-lg-12">
                    <label className="color-p mb-1">
                      <b>Documents</b>
                    </label>

                    {/* CV */}
                    {data.cv_is_required !== "Not Required" && (
                      <div className="modern-file-upload-container">
                        <div
                          className={`modern-file-upload-area ${
                            cvFileName ? "file-selected" : ""
                          }`}
                          onClick={selectCV}
                        >
                          {!cvFileName ? (
                            <div className="upload-content">
                              <div className="upload-icon">
                                <svg
                                  width="20"
                                  height="20"
                                  viewBox="0 0 24 24"
                                  fill="none"
                                  xmlns="http://www.w3.org/2000/svg"
                                >
                                  <path
                                    d="M14 2H6C4.9 2 4 2.9 4 4V20C4 21.1 4.89 22 5.99 22H18C19.1 22 20 21.1 20 20V8L14 2Z"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <polyline
                                    points="14,2 14,8 20,8"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <line
                                    x1="16"
                                    y1="13"
                                    x2="8"
                                    y2="13"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <line
                                    x1="16"
                                    y1="17"
                                    x2="8"
                                    y2="17"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <polyline
                                    points="10,9 9,9 8,9"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                </svg>
                              </div>
                              <div className="upload-text">
                                <div className="upload-title">CV is required!</div>
                                <div className="upload-subtitle">
                                  Click to attach your CV
                                </div>
                              </div>
                            </div>
                          ) : (
                            <div className="file-selected-content">
                              <div className="file-icon">
                                <svg
                                  width="20"
                                  height="20"
                                  viewBox="0 0 24 24"
                                  fill="none"
                                  xmlns="http://www.w3.org/2000/svg"
                                >
                                  <path
                                    d="M14 2H6C4.9 2 4 2.9 4 4V20C4 21.1 4.89 22 5.99 22H18C19.1 22 20 21.1 20 20V8L14 2Z"
                                    stroke="#22c55e"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <polyline
                                    points="14,2 14,8 20,8"
                                    stroke="#22c55e"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <path
                                    d="M9 13L11 15L15 11"
                                    stroke="#22c55e"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                </svg>
                              </div>
                              <div className="file-details">
                                <div className="file-name">{cvFileName}</div>
                                <div className="file-action">Click to replace</div>
                              </div>
                            </div>
                          )}
                          <input
                            ref={cvInputRef}
                            accept=".pdf"
                            onChange={onFileCV}
                            id="file-cv"
                            type="file"
                            style={{ display: "none" }}
                          />
                        </div>
                        {cvFileName && (
                          <button
                            type="button"
                            className="remove-file-btn"
                            onClick={removeCV}
                          >
                            <svg
                              width="16"
                              height="16"
                              viewBox="0 0 24 24"
                              fill="none"
                              xmlns="http://www.w3.org/2000/svg"
                            >
                              <polyline
                                points="3,6 5,6 21,6"
                                stroke="currentColor"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                              <path
                                d="M8 6V4C8 3.46957 8.21071 2.96086 8.58579 2.58579C8.96086 2.21071 9.46957 2 10 2H14C14.5304 2 15.0391 2.21071 15.4142 2.58579C15.7893 2.96086 16 3.46957 16 4V6M19 6V20C19 20.5304 18.7893 21.0391 18.4142 21.4142C18.0391 21.7893 17.5304 22 17 22H7C6.46957 22 5.96086 21.7893 5.58579 21.4142C5.21071 21.0391 5 20.5304 5 20V6H19Z"
                                stroke="currentColor"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                            </svg>
                            Remove
                          </button>
                        )}
                      </div>
                    )}

                    {/* Cover Letter */}
                    {data.caver_latter_is_required !== "Not Required" && (
                      <div className="modern-file-upload-container mt-3">
                        <div
                          className={`modern-file-upload-area ${
                            coverLetterFileName ? "file-selected" : ""
                          }`}
                          onClick={selectCoverLetter}
                        >
                          {!coverLetterFileName ? (
                            <div className="upload-content">
                              <div className="upload-icon">
                                <svg
                                  width="20"
                                  height="20"
                                  viewBox="0 0 24 24"
                                  fill="none"
                                  xmlns="http://www.w3.org/2000/svg"
                                >
                                  <path
                                    d="M14 2H6C4.9 2 4 2.9 4 4V20C4 21.1 4.89 22 5.99 22H18C19.1 22 20 21.1 20 20V8L14 2Z"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <polyline
                                    points="14,2 14,8 20,8"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <line
                                    x1="16"
                                    y1="13"
                                    x2="8"
                                    y2="13"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <line
                                    x1="16"
                                    y1="17"
                                    x2="8"
                                    y2="17"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <polyline
                                    points="10,9 9,9 8,9"
                                    stroke="currentColor"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                </svg>
                              </div>
                              <div className="upload-text">
                                <div className="upload-title">Cover Letter</div>
                                <div className="upload-subtitle">
                                  Click to attach your cover letter
                                </div>
                              </div>
                            </div>
                          ) : (
                            <div className="file-selected-content">
                              <div className="file-icon">
                                <svg
                                  width="20"
                                  height="20"
                                  viewBox="0 0 24 24"
                                  fill="none"
                                  xmlns="http://www.w3.org/2000/svg"
                                >
                                  <path
                                    d="M14 2H6C4.9 2 4 2.9 4 4V20C4 21.1 4.89 22 5.99 22H18C19.1 22 20 21.1 20 20V8L14 2Z"
                                    stroke="#22c55e"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <polyline
                                    points="14,2 14,8 20,8"
                                    stroke="#22c55e"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                  <path
                                    d="M9 13L11 15L15 11"
                                    stroke="#22c55e"
                                    strokeWidth="2"
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                  />
                                </svg>
                              </div>
                              <div className="file-details">
                                <div className="file-name">
                                  {coverLetterFileName}
                                </div>
                                <div className="file-action">Click to replace</div>
                              </div>
                            </div>
                          )}
                          <input
                            ref={coverInputRef}
                            accept=".pdf"
                            onChange={onFileCoverLetter}
                            id="file-cover-letter"
                            type="file"
                            style={{ display: "none" }}
                          />
                        </div>
                        {coverLetterFileName && (
                          <button
                            type="button"
                            className="remove-file-btn"
                            onClick={removeCoverLetter}
                          >
                            <svg
                              width="16"
                              height="16"
                              viewBox="0 0 24 24"
                              fill="none"
                              xmlns="http://www.w3.org/2000/svg"
                            >
                              <polyline
                                points="3,6 5,6 21,6"
                                stroke="currentColor"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                              <path
                                d="M8 6V4C8 3.46957 8.21071 2.96086 8.58579 2.58579C8.96086 2.21071 9.46957 2 10 2H14C14.5304 2 15.0391 2.21071 15.4142 2.58579C15.7893 2.96086 16 3.46957 16 4V6M19 6V20C19 20.5304 18.7893 21.0391 18.4142 21.4142C18.0391 21.7893 17.5304 22 17 22H7C6.46957 22 5.96086 21.7893 5.58579 21.4142C5.21071 21.0391 5 20.5304 5 20V6H19Z"
                                stroke="currentColor"
                                strokeWidth="2"
                                strokeLinecap="round"
                                strokeLinejoin="round"
                              />
                            </svg>
                            Remove
                          </button>
                        )}
                      </div>
                    )}
                  </div>
                </div>

                {/* Terms and Conditions */}
                <div className="row mt-4">
                  <div className="col-12">
                    <label className="d-flex align-items-center gap-2">
                      <input
                        type="checkbox"
                        checked={isAgree}
                        onChange={(e) => setIsAgree(e.target.checked)}
                      />
                      <span>
                        I agree to the FLOWXTRA's{" "}
                        <a href="#" className="text-primary">
                          Terms and Conditions
                        </a>{" "}
                        and{" "}
                        <a href="#" className="text-primary">
                          Privacy Policy
                        </a>
                      </span>
                    </label>
                  </div>
                </div>

                {/* Apply Now Button */}
                <div className="row mt-4">
                  <div className="col-12 d-flex justify-content-end">
                    <button
                      type="button"
                      className="btn-theme fs-16"
                      onClick={handleFormSubmit}
                      disabled={!isAgree || isSubmitting}
                    >
                      {isSubmitting ? (
                        <>
                          <span
                            className="spinner-border spinner-border-sm me-2"
                            role="status"
                            aria-hidden="true"
                          ></span>
                          Submitting...
                        </>
                      ) : hasQuestions ? (
                        "Next"
                      ) : (
                        "Apply Now"
                      )}
                    </button>
                  </div>
                </div>

                {/* Confirmation Message */}
                <div className="row mt-3">
                  <div className="col-12">
                    <p className="gray-6 text-center">
                      You will receive a link to confirm your email address, and
                      your account will be created. You can also apply by
                      creating a password.
                    </p>
                  </div>
                </div>
              </form>

              {/* Contact Person */}
              {data.job_contact_person?.length ? (
                <div className="row mt-6">
                  <div className="col-12">
                    <h2 className="fs-22 fw-bold color-dark-green mb-3">
                      Contact Person
                    </h2>
                  </div>
                  {data.job_contact_person.map((item, idx) => (
                    <div className="col-12" key={idx}>
                      <div className="d-flex justify-content-start align-items-center flex-wrap mt-3 contact-person-container">
                        <div className="me-3 mb-2 mb-md-0">
                          <Image
                            src={
                              item.user.photo || "/user-avatar-profile.svg"
                            }
                            alt="Contact person"
                            width={52}
                            height={52}
                            className="img-user"
                            unoptimized
                            onError={(e) => {
                              const target = e.target as HTMLImageElement;
                              target.src = "/user-avatar-profile.svg";
                            }}
                          />
                        </div>
                        <div className="contact-person-info">
                          <div className="fs-18 fw-bold">
                            {item.user.first_name} {item.user.last_name}
                          </div>
                          <div className="fs-16 mt-1 color-7D7D7D">
                            {item.user.email}
                          </div>
                          {item.user.position && (
                            <p className="color-7D7D7D mb-0">
                              {item.user.position}
                            </p>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : null}

              {/* Footer */}
              <div className="row mt-4">
                <div className="col-12">
                  <p className="gray-6 fs-14 text-center">
                    This site is protected by reCAPTCHA. Google{" "}
                    <a href="#" className="text-primary">
                      Privacy Policy
                    </a>{" "}
                    and{" "}
                    <a href="#" className="text-primary">
                      Terms_of_Service
                    </a>{" "}
                    apply.
                  </p>
                </div>
              </div>
            </div>
          )}

          {/* Step 2: Questions */}
          {currentStep === 2 && hasQuestions && (
            <div className="modern-application-form">
              <div className="form-header">
                <div className="header-content">
                  <div className="header-title">
                    Pre-screen Candidates
                  </div>
                </div>
              </div>

              <div className="questions-container">
                {data.job_appliction_question && data.job_appliction_question.map((q, index) => {
                  // Normalize question type - handle both formats (yes-no/yes_no, single/single_choice, etc.)
                  const rawType =q.answer_type as string;
                  const qType = rawType.toLowerCase().replace(/-/g, "_");
                  const title =
                    q.title || (q as any).name || `Question ${index + 1}`;
                  const questionText = q.question || "";

                  return (
                    <div   
                      key={q.id}
                      className="question-card"
                      data-question-type={qType}
                    >
                      <div className="question-header">
                        <div className="question-number">
                          <span>{index + 1}</span>
                        </div>
                        <div className="question-content">
                          <div className="question-title">{title}</div>
                          {q.introduction && (
                            <div
                              className="question-intro"
                              dangerouslySetInnerHTML={{
                                __html: q.introduction,
                              }}
                            />
                          )}
                          {questionText && (
                            <div className="question-text">{questionText}</div>
                          )}
                          <div
                            className={`question-type-badge type-${qType}`}
                          >
                            {getQuestionTypeName(qType)}
                          </div>
                        </div>
                        {q.required && (
                          <div className="question-required">
                            <span className="required-indicator">*</span>
                          </div>
                        )}
                      </div>

                      <div className="question-answer">
                        {(qType === TypeFieldAnswer.SHORT_TEXT || qType === "text" || qType === "short_text") && (
                          <div className="answer-input">
                            <div className="input-wrapper">
                              <input
                                className="modern-input"
                                placeholder="Enter your answer here..."
                                value={questionsAnswers[q.id] || ""}
                                onChange={(e) =>
                                  handleQuestionAnswer(q, e.target.value)
                                }
                              />
                            </div>
                          </div>
                        )}

                        {(qType === TypeFieldAnswer.NUMBER || qType === "number") && (
                          <div className="answer-input">
                            <div className="input-wrapper">
                              <input
                                type="number"
                                className="modern-input number-input"
                                placeholder="Enter a number..."
                                value={questionsAnswers[q.id] || ""}
                                onChange={(e) =>
                                  handleQuestionAnswer(q, e.target.value)
                                }
                              />
                            </div>
                          </div>
                        )}

                        {(qType === TypeFieldAnswer.DATE || qType === "date") && (
                          <div className="answer-input">
                            <div className="input-wrapper">
                              <input
                                type="date"
                                className="modern-input date-input"
                                value={questionsAnswers[q.id] || ""}
                                onChange={(e) =>
                                  handleQuestionAnswer(q, e.target.value)
                                }
                              />
                            </div>
                          </div>
                        )}

                        {(qType === TypeFieldAnswer.YES_NO || qType === "yes_no" || qType === "yes-no") && (
                          <div className="answer-input">
                            <div className="radio-group-wrapper yes-no-group">
                              <div className="modern-radio-group">
                                <div className="radio-option">
                                  <label className="modern-radio-button yes-option">
                                    <input
                                      type="radio"
                                      name={`q-${q.id}`}
                                      checked={questionsAnswers[q.id] === true || questionsAnswers[q.id] === "yes"}
                                      onChange={() =>
                                        handleQuestionAnswer(q, true)
                                      }
                                    />
                                    <svg
                                      width="16"
                                      height="16"
                                      viewBox="0 0 24 24"
                                      fill="none"
                                      xmlns="http://www.w3.org/2000/svg"
                                    >
                                      <path
                                        d="M9 12L11 14L15 10M21 12C21 16.9706 16.9706 21 12 21C7.02944 21 3 16.9706 3 12C3 7.02944 7.02944 3 12 3C16.9706 3 21 7.02944 21 12Z"
                                        stroke="currentColor"
                                        strokeWidth="2"
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                      />
                                    </svg>
                                    Yes
                                  </label>
                                </div>
                                <div className="radio-option">
                                  <label className="modern-radio-button no-option">
                                    <input
                                      type="radio"
                                      name={`q-${q.id}`}
                                      checked={questionsAnswers[q.id] === false || questionsAnswers[q.id] === "no"}
                                      onChange={() =>
                                        handleQuestionAnswer(q, false)
                                      }
                                    />
                                    <svg
                                      width="16"
                                      height="16"
                                      viewBox="0 0 24 24"
                                      fill="none"
                                      xmlns="http://www.w3.org/2000/svg"
                                    >
                                      <circle
                                        cx="12"
                                        cy="12"
                                        r="10"
                                        stroke="currentColor"
                                        strokeWidth="2"
                                      />
                                      <line
                                        x1="15"
                                        y1="9"
                                        x2="9"
                                        y2="15"
                                        stroke="currentColor"
                                        strokeWidth="2"
                                      />
                                      <line
                                        x1="9"
                                        y1="9"
                                        x2="15"
                                        y2="15"
                                        stroke="currentColor"
                                        strokeWidth="2"
                                      />
                                    </svg>
                                    No
                                  </label>
                                </div>
                              </div>
                            </div>
                          </div>
                        )}

                        {(qType === TypeFieldAnswer.SINGLE_CHOICE   ) && (
                          <div className="answer-input">
                            <div className="radio-group-wrapper">
                              <div className="modern-radio-group">
                                {(q.job_appliction_fild_answer || q.options?.map((o) => ({ id: o.id, fild: o.name })))?.map((o) => (
                                  <div key={o.id} className="radio-option">
                                    <label className="modern-radio-button">
                                      <input
                                        type="radio"
                                        name={`q-${q.id}`}
                                        value={o.fild}
                                        checked={String(questionsAnswers[q.id]) === String(o.fild)}
                                        onChange={(e) =>
                                          handleQuestionAnswer(q, e.target.value)
                                        }
                                      />
                                      {o.fild}
                                    </label>
                                  </div>
                                ))}
                              </div>
                            </div>
                          </div>
                        )}

                        {(qType === TypeFieldAnswer.MULTIPLE_CHOICE  ) && (
                          <div className="answer-input">
                            <div className="checkbox-group-wrapper">
                              <div className="modern-checkbox-group">
                                {(q.job_appliction_fild_answer || q.options?.map((o) => ({ id: o.id, fild: o.name })))?.map((o) => {
                                  const currentAnswers: string[] = questionsAnswers[q.id] || [];
                                  const isChecked = currentAnswers.includes(o.fild);
                                  return (
                                    <div key={o.id} className="checkbox-option">
                                      <label className="modern-checkbox-button">
                                        <input
                                          type="checkbox"
                                          checked={isChecked}
                                          onChange={(e) => {
                                            setQuestionsAnswers((prev) => {
                                              const current: string[] =
                                                prev[q.id] || [];
                                              if (e.target.checked) {
                                                return {
                                                  ...prev,
                                                  [q.id]: [...current, o.fild],
                                                };
                                              }
                                              return {
                                                ...prev,
                                                [q.id]: current.filter(
                                                  (x) => x !== o.fild
                                                ),
                                              };
                                            });
                                          }}
                                        />
                                        {o.fild}
                                      </label>
                                    </div>
                                  );
                                })}
                              </div>
                            </div>
                          </div>
                        )}

                        {(qType === TypeFieldAnswer.FILE || qType === "file") && (
                          <div className="answer-input">
                            <div className="modern-file-upload-container">
                              <div
                                className={`modern-file-upload-area ${hasQuestionFile(q.id) ? "file-selected" : ""}`}
                                onClick={() => selectQuestionFile(q.id)}
                              >
                                {!hasQuestionFile(q.id) ? (
                                  <div className="upload-content">
                                    <div className="upload-icon">
                                      <svg
                                        width="20"
                                        height="20"
                                        viewBox="0 0 24 24"
                                        fill="none"
                                        xmlns="http://www.w3.org/2000/svg"
                                      >
                                        <path
                                          d="M14 2H6C4.9 2 4 2.9 4 4V20C4 21.1 4.89 22 5.99 22H18C19.1 22 20 21.1 20 20V8L14 2Z"
                                          stroke="currentColor"
                                          strokeWidth="2"
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                        />
                                        <polyline
                                          points="14,2 14,8 20,8"
                                          stroke="currentColor"
                                          strokeWidth="2"
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                        />
                                        <line
                                          x1="16"
                                          y1="13"
                                          x2="8"
                                          y2="13"
                                          stroke="currentColor"
                                          strokeWidth="2"
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                        />
                                        <line
                                          x1="16"
                                          y1="17"
                                          x2="8"
                                          y2="17"
                                          stroke="currentColor"
                                          strokeWidth="2"
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                        />
                                        <polyline
                                          points="10,9 9,9 8,9"
                                          stroke="currentColor"
                                          strokeWidth="2"
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                        />
                                      </svg>
                                    </div>
                                    <div className="upload-text">
                                      <div className="upload-title">Click to upload file</div>
                                      <div className="upload-subtitle">
                                        PDF, Word, RTF, TXT, or Image files
                                      </div>
                                    </div>
                                  </div>
                                ) : (
                                  <div className="file-selected-content">
                                    <div className="file-icon">
                                      <svg
                                        width="20"
                                        height="20"
                                        viewBox="0 0 24 24"
                                        fill="none"
                                        xmlns="http://www.w3.org/2000/svg"
                                      >
                                        <path
                                          d="M14 2H6C4.9 2 4 2.9 4 4V20C4 21.1 4.89 22 5.99 22H18C19.1 22 20 21.1 20 20V8L14 2Z"
                                          stroke="#22c55e"
                                          strokeWidth="2"
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                        />
                                        <polyline
                                          points="14,2 14,8 20,8"
                                          stroke="#22c55e"
                                          strokeWidth="2"
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                        />
                                        <path
                                          d="M9 13L11 15L15 11"
                                          stroke="#22c55e"
                                          strokeWidth="2"
                                          strokeLinecap="round"
                                          strokeLinejoin="round"
                                        />
                                      </svg>
                                    </div>
                                    <div className="file-details">
                                      <div className="file-name">{questionFileNames[q.id]}</div>
                                      <div className="file-action">Click to replace</div>
                                    </div>
                                  </div>
                                )}
                                <input
                                  ref={(el) => {
                                    questionFileInputRefs.current[q.id] = el;
                                  }}
                                  type="file"
                                  accept=".pdf,.doc,.docx,.rtf,.txt,.jpg,.jpeg,.png,.gif"
                                  onChange={(e) => handleQuestionFileSelected(e, q.id)}
                                  style={{ display: "none" }}
                                />
                              </div>
                              {hasQuestionFile(q.id) && (
                                <button
                                  type="button"
                                  className="remove-file-btn"
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    removeQuestionFile(q.id);
                                  }}
                                >
                                  <svg
                                    width="16"
                                    height="16"
                                    viewBox="0 0 24 24"
                                    fill="none"
                                    xmlns="http://www.w3.org/2000/svg"
                                  >
                                    <polyline
                                      points="3,6 5,6 21,6"
                                      stroke="currentColor"
                                      strokeWidth="2"
                                      strokeLinecap="round"
                                      strokeLinejoin="round"
                                    />
                                    <path
                                      d="M8 6V4C8 3.46957 8.21071 2.96086 8.58579 2.58579C8.96086 2.21071 9.46957 2 10 2H14C14.5304 2 15.0391 2.21071 15.4142 2.58579C15.7893 2.96086 16 3.46957 16 4V6M19 6V20C19 20.5304 18.7893 21.0391 18.4142 21.4142C18.0391 21.7893 17.5304 22 17 22H7C6.46957 22 5.96086 21.7893 5.58579 21.4142C5.21071 21.0391 5 20.5304 5 20V6H19Z"
                                      stroke="currentColor"
                                      strokeWidth="2"
                                      strokeLinecap="round"
                                      strokeLinejoin="round"
                                    />
                                  </svg>
                                  Remove
                                </button>
                              )}
                            </div>
                          </div>
                        )}
                      </div>
                    </div>
                  );
                })}
              </div>

              <div className="d-flex justify-content-end gap-4 mt-6">
                <button
                  type="button"
                  className="btn-theme-outline"
                  onClick={() => setCurrentStep(1)}
                  disabled={isSubmitting}
                >
                  Back
                </button>
                <button
                  type="button"
                  className="btn-theme"
                  onClick={handleQuestionsSubmit}
                  disabled={isSubmitting}
                >
                  {isSubmitting ? (
                    <>
                      <span
                        className="spinner-border spinner-border-sm me-2"
                        role="status"
                        aria-hidden="true"
                      ></span>
                      Submitting...
                    </>
                  ) : (
                    "Apply Now"
                  )}
                </button>
              </div>

             
            </div>
          )}
        </div>
      </div>
    </div>
  );
}

